import type { Metadata } from "next";
import { createMetadata } from "@/lib/metadata";
import { getSiteInfo } from "@/lib/data";
import { getLanguage } from "@/lib/language";
import { getTheme } from "@/lib/theme";
import { getTranslations } from "@/lib/translations";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

const descriptionsByLang = {
  en: "Discover the mission of {siteName} to provide responsible betting recommendations and insights for UK players.",
  pt: "Descubra a missão do {siteName} para fornecer recomendações de apostas responsáveis e insights para jogadores do Reino Unido.",
  dk: "Opdag {siteName}s mission om at give ansvarlige bettinganbefalinger og indsigt til danske spillere.",
  fr: "Découvrez la mission de {siteName} pour fournir des recommandations de paris responsables et des informations pour les joueurs UK.",
  nl: "Ontdek de missie van {siteName} om verantwoorde gokaanbevelingen en inzichten te bieden voor UK spelers.",
  sv: "Upptäck {siteName}s uppdrag att tillhandahålla ansvarsfulla spelrekommendationer och insikter för UK-spelare.",
  es: "Descubre la misión de {siteName} para proporcionar recomendaciones de apuestas responsables e información para jugadores UK.",
} as const;

export async function generateMetadata(): Promise<Metadata> {
  const siteInfo = getSiteInfo();
  const language = await getLanguage();
  const description = descriptionsByLang[language].replace("{siteName}", siteInfo.siteName);
  
  return createMetadata({
    title: `About ${siteInfo.siteName}`,
    description,
    path: "/about",
    language,
  });
}

export default async function AboutPage() {
  const siteInfo = getSiteInfo();
  const language = await getLanguage();
  const theme = await getTheme();
  const translations = getTranslations(language);
  const colorTheme = getColorTheme(theme);
  
  const aboutContent = [
    {
      title: translations.about.mission.title,
      description: translations.about.mission.description,
      gradient: colorTheme.buttonBadge,
      iconSvg: (
        <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
          <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
        </svg>
      ),
    },
    {
      title: translations.about.compliance.title,
      description: translations.about.compliance.description,
      gradient: colorTheme.welcomeBonus,
      iconSvg: (
        <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
          <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
        </svg>
      ),
    },
    {
      title: translations.about.dataDriven.title,
      description: translations.about.dataDriven.description,
      gradient: colorTheme.buttonBadge,
      iconSvg: (
        <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
          <path strokeLinecap="round" strokeLinejoin="round" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
        </svg>
      ),
    },
  ];

  return (
    <div className="space-y-6 sm:space-y-8">
      <section 
        className="relative overflow-hidden rounded-2xl border px-6 py-8 text-white shadow-2xl sm:rounded-3xl sm:px-8 sm:py-10"
        style={{
          borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
          boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
        }}
      >
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
          }}
        />
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
          }}
        />
        <div className="relative space-y-4 text-center sm:space-y-5">
          <span 
            className="inline-flex items-center gap-2 rounded-full border px-3 py-1.5 text-[10px] font-semibold uppercase tracking-[0.15em] backdrop-blur-sm sm:px-4 sm:py-2 sm:text-xs"
            style={{
              borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              color: hexToRgba(colorTheme.buttonBadge, 0.9),
            }}
          >
            <span className="relative flex h-1.5 w-1.5 sm:h-2 sm:w-2">
              <span 
                className="absolute inline-flex h-full w-full animate-ping rounded-full opacity-75" 
                style={{ backgroundColor: colorTheme.buttonBadge }}
              />
              <span 
                className="relative inline-flex h-full w-full rounded-full" 
                style={{ backgroundColor: colorTheme.buttonBadge }}
              />
            </span>
            {translations.about.badge}
          </span>
          <h1 className="text-3xl font-bold leading-[1.1] tracking-tight text-white sm:text-5xl lg:text-6xl">
            {translations.about.title.replace("{siteName}", siteInfo.siteName)}
          </h1>
          <p className="mx-auto max-w-3xl text-sm leading-relaxed text-white/90 sm:text-lg">
            {translations.about.subtitle.replace("{siteName}", siteInfo.siteName)}
          </p>
        </div>
      </section>

      <section className="grid gap-4 sm:grid-cols-3 sm:gap-6">
        {aboutContent.map((item) => (
          <article
            key={item.title}
            className="group relative flex min-h-[200px] flex-col overflow-hidden rounded-xl border p-5 text-center shadow-lg transition-all duration-300 ease-out hover:-translate-y-1 sm:min-h-[220px] sm:rounded-2xl sm:p-6"
            style={{
              borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
              background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
              boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
            }}
          >
            <div 
              className="absolute inset-0 transition-all duration-300" 
              style={{
                background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
              }}
            />
            <div 
              className="relative z-10 mx-auto mb-3 flex h-12 w-12 items-center justify-center rounded-xl text-white shadow-lg transition-transform duration-300 group-hover:scale-110 sm:mb-4 sm:h-14 sm:w-14"
              style={{
                background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
              }}
            >
              {item.iconSvg}
            </div>
            <h2 className="relative z-10 mb-2.5 text-base font-extrabold text-white sm:mb-3 sm:text-lg">
              {item.title}
            </h2>
            <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">{item.description}</p>
          </article>
        ))}
      </section>

      <section 
        className="relative space-y-5 overflow-hidden rounded-2xl border p-5 shadow-xl backdrop-blur-sm sm:rounded-3xl sm:space-y-6 sm:p-8"
        style={{
          borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
          boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
        }}
      >
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
          }}
        />
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
          }}
        />
        <div className="relative space-y-3 text-center sm:space-y-4">
          <h2 className="relative z-10 text-xl font-bold tracking-tight text-white sm:text-3xl">
            {translations.about.commitment.title}
          </h2>
          <p className="relative z-10 mx-auto max-w-2xl text-sm leading-relaxed text-white/80 sm:text-base">
            {translations.about.commitment.description}
          </p>
        </div>
        <div className="relative grid gap-4 sm:gap-5 md:grid-cols-2">
          <div 
            className="group relative flex min-h-[200px] flex-col overflow-hidden rounded-xl border p-5 text-center shadow-lg transition-all duration-300 ease-out hover:-translate-y-1 sm:min-h-[220px] sm:rounded-2xl sm:p-6"
            style={{
              borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
              background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
              boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
            }}
          >
            <div 
              className="absolute inset-0 transition-all duration-300" 
              style={{
                background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
              }}
            />
            <div 
              className="relative z-10 mx-auto mb-3 flex h-12 w-12 items-center justify-center rounded-xl text-white shadow-lg transition-transform duration-300 group-hover:scale-110 sm:mb-4 sm:h-14 sm:w-14"
              style={{
                background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
              }}
            >
              <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                <path strokeLinecap="round" strokeLinejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
              </svg>
            </div>
            <h3 className="relative z-10 mb-2.5 text-base font-extrabold text-white sm:mb-3 sm:text-lg">
              {translations.about.trust.title}
            </h3>
            <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
              {translations.about.trust.description}
            </p>
          </div>
          <div 
            className="group relative flex min-h-[200px] flex-col overflow-hidden rounded-xl border p-5 text-center shadow-lg transition-all duration-300 ease-out hover:-translate-y-1 sm:min-h-[220px] sm:rounded-2xl sm:p-6"
            style={{
              borderColor: hexToRgba(colorTheme.welcomeBonus, 0.25),
              background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
              boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.welcomeBonus, 0.1)}`,
            }}
          >
            <div 
              className="absolute inset-0 transition-all duration-300" 
              style={{
                background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.welcomeBonus, 0)}, ${hexToRgba(colorTheme.welcomeBonus, 0)}, ${hexToRgba(colorTheme.welcomeBonus, 0)})`,
              }}
            />
            <div 
              className="relative z-10 mx-auto mb-3 flex h-12 w-12 items-center justify-center rounded-xl text-white shadow-lg transition-transform duration-300 group-hover:scale-110 sm:mb-4 sm:h-14 sm:w-14"
              style={{
                background: `linear-gradient(to bottom right, ${colorTheme.welcomeBonus}, ${colorTheme.welcomeBonus})`,
                boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.welcomeBonus, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
              }}
            >
              <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M13 10V3L4 14h7v7l9-11h-7z" />
              </svg>
            </div>
            <h3 className="relative z-10 mb-2.5 text-base font-extrabold text-white sm:mb-3 sm:text-lg">
              {translations.about.performance.title}
            </h3>
            <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
              {translations.about.performance.description}
            </p>
          </div>
        </div>
      </section>

      <section 
        className="relative space-y-5 overflow-hidden rounded-2xl border p-5 shadow-xl backdrop-blur-sm sm:rounded-3xl sm:space-y-6 sm:p-8"
        style={{
          borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
          boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
        }}
      >
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
          }}
        />
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
          }}
        />
        <div className="relative space-y-4">
          <h2 className="text-center text-xl font-bold tracking-tight text-white sm:text-3xl">
            Our Core Values
          </h2>
          <div className="grid gap-4 sm:grid-cols-2 sm:gap-5">
            <div 
              className="group relative flex flex-col overflow-hidden rounded-xl border p-5 shadow-lg transition-all duration-300 ease-out sm:rounded-2xl sm:p-6"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
                background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
                boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
                }}
              />
              <div className="relative z-10 mb-3 flex items-center gap-3">
                <div 
                  className="flex h-11 w-11 shrink-0 items-center justify-center rounded-lg text-white shadow-md transition-transform duration-300 group-hover:scale-110 sm:h-12 sm:w-12"
                  style={{
                    background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                    boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                  }}
                >
                  <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
                  </svg>
                </div>
                <h3 className="text-base font-extrabold text-white sm:text-lg">Excellence</h3>
              </div>
              <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
                We maintain the highest standards in partner selection, ensuring every recommendation meets rigorous quality criteria for licensing, user experience, and compliance.
              </p>
            </div>
            <div 
              className="group relative flex flex-col overflow-hidden rounded-xl border p-5 shadow-lg transition-all duration-300 ease-out sm:rounded-2xl sm:p-6"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.5),
                background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.cardBackground, 0.9)}, ${hexToRgba(colorTheme.cardBackground, 0.85)}, ${hexToRgba(colorTheme.cardBackground, 0.9)})`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
                }}
              />
              <div className="relative z-10 mb-3 flex items-center gap-3">
                <div 
                  className="flex h-11 w-11 shrink-0 items-center justify-center rounded-lg text-white shadow-md transition-transform duration-300 group-hover:scale-110 sm:h-12 sm:w-12"
                  style={{
                    background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                    boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                  }}
                >
                  <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z" />
                  </svg>
                </div>
                <h3 className="text-base font-extrabold text-white sm:text-lg">User-Centric</h3>
              </div>
              <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
                Your betting experience is our priority. We continuously optimize our platform based on user feedback and performance data to deliver the best possible experience.
              </p>
            </div>
            <div 
              className="group relative flex flex-col overflow-hidden rounded-xl border p-5 shadow-lg transition-all duration-300 ease-out sm:rounded-2xl sm:p-6"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.5),
                background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.cardBackground, 0.9)}, ${hexToRgba(colorTheme.cardBackground, 0.85)}, ${hexToRgba(colorTheme.cardBackground, 0.9)})`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
                }}
              />
              <div className="relative z-10 mb-3 flex items-center gap-3">
                <div 
                  className="flex h-11 w-11 shrink-0 items-center justify-center rounded-lg text-white shadow-md transition-transform duration-300 group-hover:scale-110 sm:h-12 sm:w-12"
                  style={{
                    background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                    boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                  }}
                >
                  <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                  </svg>
                </div>
                <h3 className="text-base font-extrabold text-white sm:text-lg">Security First</h3>
              </div>
              <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
                We prioritize your data security and privacy. All information is handled with the utmost care, following GDPR guidelines and industry best practices.
              </p>
            </div>
            <div 
              className="group relative flex flex-col overflow-hidden rounded-xl border p-5 shadow-lg transition-all duration-300 ease-out sm:rounded-2xl sm:p-6"
              style={{
                borderColor: hexToRgba(colorTheme.welcomeBonus, 0.25),
                background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
                boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.welcomeBonus, 0.1)}`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.welcomeBonus, 0)}, ${hexToRgba(colorTheme.welcomeBonus, 0)}, ${hexToRgba(colorTheme.welcomeBonus, 0)})`,
                }}
              />
              <div className="relative z-10 mb-3 flex items-center gap-3">
                <div 
                  className="flex h-11 w-11 shrink-0 items-center justify-center rounded-lg text-white shadow-md transition-transform duration-300 group-hover:scale-110 sm:h-12 sm:w-12"
                  style={{
                    background: `linear-gradient(to bottom right, ${colorTheme.welcomeBonus}, ${colorTheme.welcomeBonus})`,
                    boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.welcomeBonus, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                  }}
                >
                  <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                  </svg>
                </div>
                <h3 className="text-base font-extrabold text-white sm:text-lg">Innovation</h3>
              </div>
              <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
                We stay ahead of industry trends, continuously improving our platform with cutting-edge technology and innovative features to enhance your betting journey.
              </p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
}
