"use client";

import { useEffect, useState, useMemo, useCallback } from "react";
import Image from "next/image";
import type { Partner } from "@/types/data";
import type { Translations } from "@/lib/translations";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface PartnerCardProps {
  readonly partner: Partner;
  readonly affiliateHref: string;
  readonly index?: number;
  readonly translations: Translations;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

function getPositionLabel(index: number, positionLabels: readonly string[]): string {
  if (index < positionLabels.length) {
    return positionLabels[index];
  }
  // Fallback for positions beyond the provided labels
  // This is a generic fallback - ideally all positions should be in translations
  const fallbackSuffixes: Record<number, string> = { 1: "st", 2: "nd", 3: "rd" };
  const suffix = fallbackSuffixes[index % 10] || "th";
  return `${index + 1}${suffix} Option`;
}

/**
 * Gets badge color based on position index and theme
 * Uses first color (buttonBadge) with variations for different positions
 */
function getBadgeColors(index: number, theme: ThemeId): { fromColor: string; toColor: string } {
  const colorTheme = getColorTheme(theme);
  
  // Create variations of the buttonBadge color for different positions
  // Use lighter/darker variations of the first color
  const baseColor = colorTheme.buttonBadge;
  
  // Helper to lighten/darken color
  const lighten = (hex: string, percent: number) => {
    const num = parseInt(hex.replace("#", ""), 16);
    const r = Math.min(255, (num >> 16) + Math.round(255 * percent));
    const g = Math.min(255, ((num >> 8) & 0x00FF) + Math.round(255 * percent));
    const b = Math.min(255, (num & 0x0000FF) + Math.round(255 * percent));
    return `#${((r << 16) | (g << 8) | b).toString(16).padStart(6, '0')}`;
  };
  
  const variations = [
    { from: baseColor, to: lighten(baseColor, 0.15) },      // Index 0 - Top Choice (lighter)
    { from: lighten(baseColor, -0.1), to: baseColor },        // Index 1 - Editor's Choice (darker to base)
    { from: baseColor, to: lighten(baseColor, -0.1) },        // Index 2 - Best Match (base to darker)
    { from: lighten(baseColor, 0.1), to: lighten(baseColor, -0.05) }, // Index 3 - Fourth
    { from: lighten(baseColor, -0.15), to: lighten(baseColor, 0.05) }, // Index 4+ - Fifth and others
  ];
  
  const variation = variations[index % variations.length];
  return {
    fromColor: variation.from,
    toColor: variation.to,
  };
}

export function PartnerCard({ partner, affiliateHref, index = 0, translations, language, theme }: PartnerCardProps) {
  const [isVisible, setIsVisible] = useState(false);

  // Get color theme based on theme
  const colorTheme = useMemo(() => getColorTheme(theme), [theme]);

  // Memoize animation delay to prevent recalculation
  const animationDelay = useMemo(() => index * 50, [index]);

  useEffect(() => {
    const timer = setTimeout(() => {
      setIsVisible(true);
    }, animationDelay);
    return () => clearTimeout(timer);
  }, [animationDelay]);
  
  // Memoize position label using translations
  const positionLabel = useMemo(
    () => getPositionLabel(index, translations.positionLabels),
    [index, translations.positionLabels]
  );

  // Memoize badge colors based on position and theme
  const badgeColors = useMemo(
    () => getBadgeColors(index, theme),
    [index, theme]
  );

  // Generate vote count based on partner name (random but consistent per partner)
  const voteCount = useMemo(() => {
    // Create a simple hash from partner name for consistent "random" value
    let hash = 0;
    for (let i = 0; i < partner.name.length; i++) {
      const char = partner.name.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32-bit integer
    }
    // Generate a number between 1.5k and 5k based on hash
    const baseCount = 1500 + (Math.abs(hash) % 3500); // Range: 1500-5000
    if (baseCount >= 1000) {
      const kValue = (baseCount / 1000).toFixed(1);
      // Remove .0 if it's a whole number
      return kValue.endsWith('.0') ? `${kValue.replace('.0', '')}k` : `${kValue}k`;
    }
    return `${baseCount}+`;
  }, [partner.name]);

  // Memoize click handler to prevent recreation on each render
  const handleClick = useCallback(async () => {
    // Track click asynchronously without blocking navigation
    try {
      await fetch('/api/clicks', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ partnerName: partner.name }),
      });
    } catch (error) {
      // Silently fail - don't block user navigation
      console.error('Failed to track click:', error);
    }
  }, [partner.name]);

  return (
    <article
      className={`group relative flex flex-col overflow-hidden transition-all duration-500 ease-out ${
        isVisible
          ? "translate-y-0 opacity-100"
          : "translate-y-4 opacity-0"
      } hover:-translate-y-3 hover:scale-[1.02] focus-within:-translate-y-3 focus-within:scale-[1.02] active:scale-[0.99]`}
      style={{
        padding: '3px',
        borderRadius: '1rem',
        boxShadow: `0 4px 12px 0 ${hexToRgba(colorTheme.cardBackground, 0.3)}, 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
      }}
      onMouseEnter={(e) => {
        e.currentTarget.style.boxShadow = `0 24px 48px -12px ${hexToRgba(colorTheme.buttonBadge, 0.2)}, 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
      }}
      onMouseLeave={(e) => {
        e.currentTarget.style.boxShadow = `0 4px 12px 0 ${hexToRgba(colorTheme.cardBackground, 0.3)}, 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`;
      }}
      onFocus={(e) => {
        e.currentTarget.style.boxShadow = `0 24px 48px -12px ${hexToRgba(colorTheme.buttonBadge, 0.2)}, 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
      }}
      onBlur={(e) => {
        e.currentTarget.style.boxShadow = `0 4px 12px 0 ${hexToRgba(colorTheme.cardBackground, 0.3)}, 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`;
      }}
    >
      {/* Striking animated border gradient - outer layer */}
      <div 
        className="absolute opacity-80 sm:opacity-90"
        style={{
          top: '-3px',
          left: '-3px',
          right: '-3px',
          bottom: '-3px',
          backgroundImage: `linear-gradient(90deg, ${hexToRgba(colorTheme.buttonBadge, 0.3)}, ${hexToRgba(colorTheme.buttonBadge, 0.5)}, ${hexToRgba(colorTheme.buttonBadge, 0.3)}, ${hexToRgba(colorTheme.buttonBadge, 0.5)})`,
          backgroundSize: '200% 100%',
          animation: 'shimmer-border 3s ease-in-out infinite',
          borderRadius: '1rem',
          zIndex: 0,
        }}
      />
      
      {/* Inner content container */}
      <div 
        className="relative z-10 h-full w-full p-1.5 sm:p-4" 
        style={{ 
          borderRadius: 'calc(1rem - 3px)',
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
        }}
      >
        {/* Sparkling background effects - subtle buttonBadge accents */}
      <div 
        className="absolute inset-0 sm:opacity-0 transition-all duration-500 group-hover:opacity-100 z-0" 
        style={{
          background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0.1)}, ${hexToRgba(colorTheme.buttonBadge, 0.05)}, ${hexToRgba(colorTheme.buttonBadge, 0.1)})`,
        }}
      />
      
      {/* Outer glow effect - buttonBadge color */}
      <div 
        className="absolute -inset-1 rounded-2xl sm:opacity-0 blur-xl transition-all duration-500 group-hover:opacity-100 sm:rounded-3xl z-0" 
        style={{
          background: `linear-gradient(to right, ${hexToRgba(colorTheme.buttonBadge, 0.15)}, ${hexToRgba(colorTheme.buttonBadge, 0.12)}, ${hexToRgba(colorTheme.buttonBadge, 0.15)})`,
        }}
      />
      
      {/* Sparkle particles - buttonBadge color */}
      <div 
        className="absolute top-3 left-3 h-1 w-1 rounded-full sm:opacity-0 transition-all duration-500 group-hover:opacity-100 animate-pulse z-0" 
        style={{ 
          backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.6),
          boxShadow: `0 0 6px ${colorTheme.buttonBadge}`,
          animationDelay: `${index * 100}ms` 
        }} 
      />
      <div 
        className="absolute top-6 right-4 h-1.5 w-1.5 rounded-full sm:opacity-0 transition-all duration-500 group-hover:opacity-100 animate-pulse z-0" 
        style={{ 
          backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.6),
          boxShadow: `0 0 8px ${colorTheme.buttonBadge}`,
          animationDelay: `${index * 150}ms` 
        }} 
      />
      <div 
        className="absolute bottom-16 left-1/2 h-1 w-1 rounded-full sm:opacity-0 transition-all duration-500 group-hover:opacity-100 animate-pulse z-0" 
        style={{ 
          backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.6),
          boxShadow: `0 0 6px ${colorTheme.buttonBadge}`,
          animationDelay: `${index * 200}ms` 
        }} 
      />
      <div 
        className="absolute top-1/2 right-3 h-1 w-1 rounded-full sm:opacity-0 transition-all duration-500 group-hover:opacity-100 animate-pulse z-0" 
        style={{ 
          backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.5),
          boxShadow: `0 0 5px ${colorTheme.buttonBadge}`,
          animationDelay: `${index * 250}ms` 
        }} 
      />
      <div 
        className="absolute bottom-8 right-1/4 h-1 w-1 rounded-full sm:opacity-0 transition-all duration-500 group-hover:opacity-100 animate-pulse z-0" 
        style={{ 
          backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.5),
          boxShadow: `0 0 5px ${colorTheme.buttonBadge}`,
          animationDelay: `${index * 300}ms` 
        }} 
      />
      
        {/* Magnificent shimmer overlay - continuous animation */}
        <div 
          className="absolute inset-0 -translate-x-full animate-shimmer sm:opacity-0 z-0" 
          style={{
            background: `linear-gradient(to right, transparent, ${hexToRgba(colorTheme.buttonBadge, 0.1)}, ${hexToRgba(colorTheme.buttonBadge, 0.08)}, transparent)`,
          }}
        />
        
        {/* Secondary shimmer layer for depth */}
        <div 
          className="absolute inset-0 translate-x-full z-0" 
          style={{
            background: `linear-gradient(to left, transparent, ${hexToRgba(colorTheme.buttonBadge, 0.08)}, transparent)`,
            animation: 'shimmer 4s ease-in-out infinite reverse',
            animationDelay: '1.5s',
          }} 
        />
        
        {/* Additional accent layer */}
        <div 
          className="absolute inset-0 sm:opacity-0 transition-opacity duration-500 group-hover:opacity-100 z-0" 
          style={{
            background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0.03)}, transparent, ${hexToRgba(colorTheme.buttonBadge, 0.03)})`,
          }} 
        />

      {/* Combined color-coded banner - spans full width */}
      <div className="absolute top-0 left-0 right-0 z-20">
        <div 
          className="relative px-2 py-1 sm:px-3 sm:py-1.5" 
          style={{ 
            borderTopLeftRadius: 'calc(1rem - 3px)', 
            borderTopRightRadius: 'calc(1rem - 3px)',
            background: `linear-gradient(to right, ${badgeColors.fromColor}, ${badgeColors.toColor})`,
          }}
        >
          <div className="absolute inset-0 bg-gradient-to-r from-white/25 via-transparent to-white/25" style={{ borderTopLeftRadius: 'calc(1rem - 3px)', borderTopRightRadius: 'calc(1rem - 3px)' }} />
          <div className="relative flex items-center justify-between">
            {/* Position label - left side with transparent background */}
            <span className="rounded bg-white/35 backdrop-blur-md px-2 py-0.5 text-[9px] font-bold uppercase text-white drop-shadow-[0_2px_4px_rgba(0,0,0,0.8)] sm:px-2.5 sm:py-1 sm:text-[11px]">{positionLabel}</span>
            {/* Rating badge - right side with transparent background */}
            <div className="flex items-center gap-1 rounded bg-white/50 backdrop-blur-md px-2 py-0.5 sm:gap-1.5 sm:px-2.5 sm:py-1" style={{ boxShadow: '0 2px 8px rgba(0,0,0,0.4)' }}>
              {/* Star icons - rating normalized to 5 stars (rating/10 * 5) */}
              <div className="flex items-center gap-0.5">
                {[1, 2, 3, 4, 5].map((star) => {
                  const normalizedRating = (partner.rating / 10) * 5; // Convert 0-10 scale to 0-5 scale
                  const starFill = Math.max(0, Math.min(1, normalizedRating - (star - 1)));
                  const fillPercentage = starFill * 100;
                  const isFilled = starFill >= 1;
                  const isEmpty = starFill <= 0;
                  
                  return (
                    <svg key={star} className="h-2.5 w-2.5 flex-shrink-0 drop-shadow-[0_1px_3px_rgba(0,0,0,0.9)] sm:h-3 sm:w-3" viewBox="0 0 20 20" aria-hidden="true">
                      <defs>
                        <mask id={`starMask-${index}-${star}`}>
                          <rect x="0" y="0" width="20" height="20" fill="black" />
                          <rect x="0" y="0" width={`${fillPercentage * 0.2}`} height="20" fill="white" />
                        </mask>
                      </defs>
                      {/* Empty star background (gray) */}
                      <path fill="#9ca3af" opacity="0.5" d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                      {/* Filled star (gold) - shown fully if filled, masked if partial */}
                      {!isEmpty && (
                        <path 
                          fill="#fbbf24" 
                          mask={isFilled ? undefined : `url(#starMask-${index}-${star})`}
                          d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" 
                        />
                      )}
                    </svg>
                  );
                })}
              </div>
              <div className="flex items-center gap-1">
                <span className="text-[10px] font-extrabold text-white drop-shadow-[0_2px_6px_rgba(0,0,0,0.95)] sm:text-[12px]">{partner.rating.toFixed(1)}/10</span>
                <span className="text-[9px] font-bold text-white drop-shadow-[0_2px_4px_rgba(0,0,0,0.9)] sm:text-[11px]">({voteCount}+ votes)</span>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Logo and welcome bonus section - horizontal layout */}
      <div className="relative z-20 mt-7 flex items-center gap-2 sm:mt-11 sm:gap-3">
        {/* Logo section - on the left */}
        <div className="relative flex-shrink-0">
          <div 
            className="relative h-20 w-20 overflow-hidden rounded-xl border-2 bg-white p-1 shadow-md sm:shadow-lg transition-all duration-500 group-hover:scale-110 group-hover:shadow-xl sm:h-32 sm:w-32 sm:rounded-3xl sm:p-3"
            style={{
              borderColor: hexToRgba(colorTheme.buttonBadge, 0.6),
              boxShadow: `0 4px 6px -1px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
            }}
            onMouseEnter={(e) => {
              e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.8);
              e.currentTarget.style.boxShadow = `0 20px 25px -5px ${hexToRgba(colorTheme.buttonBadge, 0.4)}`;
            }}
            onMouseLeave={(e) => {
              e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.6);
              e.currentTarget.style.boxShadow = `0 4px 6px -1px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
            }}
          >
            {/* Sparkle effect around logo - buttonBadge color */}
            <div 
              className="absolute -inset-2 rounded-xl sm:opacity-0 blur-md transition-all duration-500 group-hover:opacity-100 sm:rounded-3xl" 
              style={{
                background: `linear-gradient(to right, ${hexToRgba(colorTheme.buttonBadge, 0.15)}, ${hexToRgba(colorTheme.buttonBadge, 0.10)}, ${hexToRgba(colorTheme.buttonBadge, 0.15)})`,
              }}
            />
            <Image
              src={partner.logo}
              alt={`${partner.name} logo`}
              fill
              className="relative z-10 object-contain brightness-100 contrast-100"
              sizes="(max-width: 640px) 80px, 128px"
              priority={index < 2}
              loading={index < 2 ? "eager" : "lazy"}
              quality={85}
              unoptimized={partner.logo.startsWith('/logos/')}
              fetchPriority={index < 2 ? "high" : "low"}
            />
          </div>
        </div>

        {/* Welcome bonus content - next to logo */}
        <div className="relative z-10 flex flex-col justify-center">
          <div 
            className="group/bonus relative inline-block overflow-hidden rounded-lg border backdrop-blur-md px-2.5 py-1.5 shadow-lg sm:px-3 sm:py-2 sm:shadow-xl"
            style={{
              borderColor: hexToRgba(colorTheme.welcomeBonus, 0.25),
              background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
              boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.welcomeBonus, 0.1)}`,
            }}
          >
            {/* Animated glitter particles */}
            <div className="absolute inset-0 opacity-60">
              <div 
                className="absolute top-1 left-1/4 h-0.5 w-0.5 rounded-full animate-pulse" 
                style={{ 
                  backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.8),
                  animationDelay: '0s' 
                }} 
              />
              <div 
                className="absolute top-2 right-1/3 h-0.5 w-0.5 rounded-full animate-pulse" 
                style={{ 
                  backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.8),
                  animationDelay: '0.5s' 
                }} 
              />
              <div 
                className="absolute bottom-1 left-1/2 h-0.5 w-0.5 rounded-full animate-pulse" 
                style={{ 
                  backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.8),
                  animationDelay: '1s' 
                }} 
              />
              <div 
                className="absolute top-1/2 right-1/4 h-0.5 w-0.5 rounded-full animate-pulse" 
                style={{ 
                  backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.8),
                  animationDelay: '1.5s' 
                }} 
              />
            </div>
            
            {/* Shimmer effect */}
            <div 
              className="absolute inset-0 -translate-x-full animate-shimmer" 
              style={{
                background: `linear-gradient(to right, transparent, ${hexToRgba(colorTheme.welcomeBonus, 0.3)}, transparent)`,
              }}
            />
            
            {/* Glowing border animation */}
            <div 
              className="absolute inset-0 rounded-lg opacity-50 animate-pulse" 
              style={{
                border: `1px solid ${hexToRgba(colorTheme.welcomeBonus, 0.4)}`,
              }}
            />
            
            {/* Content */}
            <div className="relative">
              <p className="text-xs font-bold leading-tight mb-1 sm:text-sm sm:mb-1.5 text-left tracking-wide drop-shadow-[0_2px_4px_rgba(0,0,0,0.9)] text-yellow-50">
                {translations.partners.welcomeBonus}
              </p>
              <p className="text-sm font-extrabold leading-tight text-white drop-shadow-[0_2px_6px_rgba(0,0,0,0.8)] sm:text-lg sm:leading-snug">
                {partner.welcomeBonus}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* CTA Button - highly attractive with casino/betting vibe */}
      <div className="relative z-20 mt-3 space-y-0.5 sm:mt-4 sm:space-y-1.5">
        <a
          href={affiliateHref}
          target="_blank"
          rel="nofollow sponsored noopener"
          onClick={handleClick}
          className="group/btn relative flex w-full items-center justify-center gap-2 overflow-hidden rounded-xl px-4 py-3 text-sm font-extrabold text-white transition-all duration-300 hover:scale-[1.05] active:scale-[0.96] focus-visible:outline focus-visible:outline-3 focus-visible:outline-offset-2 sm:rounded-2xl sm:px-6 sm:py-4 sm:text-lg sm:gap-2.5"
          onFocus={(e) => {
            e.currentTarget.style.outlineColor = colorTheme.buttonBadge;
          }}
          onBlur={(e) => {
            e.currentTarget.style.outlineColor = '';
          }}
          style={{
            background: `linear-gradient(to right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
            boxShadow: `0 4px 20px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, 0 2px 8px ${hexToRgba(colorTheme.cardBackground, 0.3)}`,
          }}
          onMouseEnter={(e) => {
            // Lighten the buttonBadge color on hover
            const lighten = (hex: string, percent: number) => {
              const num = parseInt(hex.replace("#", ""), 16);
              const r = Math.min(255, (num >> 16) + Math.round(255 * percent));
              const g = Math.min(255, ((num >> 8) & 0x00FF) + Math.round(255 * percent));
              const b = Math.min(255, (num & 0x0000FF) + Math.round(255 * percent));
              return `#${((r << 16) | (g << 8) | b).toString(16).padStart(6, '0')}`;
            };
            const lighter = lighten(colorTheme.buttonBadge, 0.1);
            e.currentTarget.style.background = `linear-gradient(to right, ${lighter}, ${colorTheme.buttonBadge})`;
            e.currentTarget.style.boxShadow = `0 8px 32px ${hexToRgba(colorTheme.buttonBadge, 0.6)}, 0 4px 16px ${hexToRgba(colorTheme.cardBackground, 0.4)}`;
          }}
          onMouseLeave={(e) => {
            e.currentTarget.style.background = `linear-gradient(to right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`;
            e.currentTarget.style.boxShadow = `0 4px 20px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, 0 2px 8px ${hexToRgba(colorTheme.cardBackground, 0.3)}`;
          }}
        >
          {/* Outer glow effect - bright and visible */}
          <span 
            className="absolute -inset-1 rounded-xl blur-xl opacity-60 transition-all duration-300 group-hover/btn:opacity-80 group-hover/btn:blur-2xl group-active/btn:opacity-40 sm:rounded-2xl" 
            style={{
              background: `linear-gradient(to right, ${hexToRgba(colorTheme.buttonBadge, 0.5)}, ${hexToRgba(colorTheme.buttonBadge, 0.5)}, ${hexToRgba(colorTheme.buttonBadge, 0.5)})`,
            }}
          />
          
          {/* Inner highlight glow */}
          <span 
            className="absolute inset-0 rounded-xl opacity-50 transition-opacity duration-300 group-hover/btn:opacity-70 sm:rounded-2xl" 
            style={{
              background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0.2)}, transparent, transparent)`,
            }}
          />
          
          {/* Shimmer shine effect */}
          <span 
            className="absolute inset-0 -translate-x-full transition-transform duration-1000 group-hover/btn:translate-x-full sm:rounded-2xl" 
            style={{
              background: `linear-gradient(to right, transparent, ${hexToRgba(colorTheme.buttonBadge, 0.4)}, transparent)`,
            }}
          />
          
          {/* Sparkle particles - always visible and animated */}
          <span 
            className="absolute top-2 left-4 h-1.5 w-1.5 rounded-full animate-pulse" 
            style={{ 
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.9),
              boxShadow: `0 0 8px ${colorTheme.buttonBadge}`,
              animationDelay: '0s' 
            }} 
          />
          <span 
            className="absolute top-2 right-4 h-1.5 w-1.5 rounded-full animate-pulse" 
            style={{ 
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.9),
              boxShadow: `0 0 8px ${colorTheme.buttonBadge}`,
              animationDelay: '0.3s' 
            }} 
          />
          <span 
            className="absolute bottom-2 left-1/3 h-1 w-1 rounded-full animate-pulse" 
            style={{ 
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.8),
              boxShadow: `0 0 6px ${colorTheme.buttonBadge}`,
              animationDelay: '0.6s' 
            }} 
          />
          <span 
            className="absolute bottom-2 right-1/4 h-1 w-1 rounded-full animate-pulse" 
            style={{ 
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.8),
              boxShadow: `0 0 6px ${colorTheme.buttonBadge}`,
              animationDelay: '0.9s' 
            }} 
          />
          
          {/* Content */}
          <span className="relative z-10 flex items-center gap-1.5 sm:gap-2">
            <span className="text-sm font-black text-white drop-shadow-[0_3px_10px_rgba(0,0,0,0.9)] sm:text-lg">{translations.partners.claimNow}</span>
            <svg
              className="h-3.5 w-3.5 transition-all duration-300 group-hover/btn:translate-x-2 group-hover/btn:scale-110 sm:h-5 sm:w-5"
              fill="none"
              viewBox="0 0 24 24"
              stroke="currentColor"
              strokeWidth={3}
            >
              <path strokeLinecap="round" strokeLinejoin="round" d="M13 7l5 5m0 0l-5 5m5-5H6" />
            </svg>
          </span>
        </a>
        <p className="text-center text-[8px] font-medium leading-tight text-slate-300 sm:text-xs sm:leading-relaxed">
          {translations.partners.terms}
        </p>
      </div>
      </div>
    </article>
  );
}
