import type { SiteInfo } from "@/types/data";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface FooterPTProps {
  readonly siteInfo: SiteInfo;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

export function FooterPT({ siteInfo, language, theme }: FooterPTProps) {
  const colorTheme = getColorTheme(theme);
  return (
    <footer 
      className="relative border-t"
      style={{
        borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
        background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
        boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
      }}
    >
      {/* Subtle background gradients - theme-based */}
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
        }}
      />
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
        }}
      />
      
      {/* Animated grid pattern */}
      <div className="absolute inset-0 bg-[linear-gradient(rgba(255,255,255,0.02)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,0.02)_1px,transparent_1px)] bg-[size:30px_30px] opacity-40" />
      
      <div className="relative mx-auto max-w-6xl px-4 py-6 sm:px-6 sm:py-12 lg:py-16">
        <div className="grid gap-6 sm:gap-8 md:grid-cols-2 lg:grid-cols-4 mb-8 sm:mb-12">
          {/* Contact */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Contacto</h3>
            <div className="space-y-2 sm:space-y-3 text-xs sm:text-sm">
              <p>
                <span className="font-semibold text-white/80">Email:</span>{" "}
                <a
                  href={`mailto:${siteInfo.email}`}
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  {siteInfo.email}
                </a>
              </p>
              <p>
                <span className="font-semibold text-white/80">Telefone:</span>{" "}
                <a
                  href={`tel:${siteInfo.phone.replace(/\s/g, "")}`}
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  {siteInfo.phone}
                </a>
              </p>
              <p>
                <span className="font-semibold text-white/80">Endereço:</span>{" "}
                <span className="text-white/70">{siteInfo.address}</span>
              </p>
            </div>
          </div>

          {/* Quick Links */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Links Rápidos</h3>
            <ul className="space-y-2 text-xs sm:text-sm">
              <li>
                <a
                  href="/"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Início
                </a>
              </li>
              <li>
                <a
                  href="/about"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Sobre Nós
                </a>
              </li>
              <li>
                <a
                  href="/contact"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Contacto
                </a>
              </li>
            </ul>
          </div>

          {/* Legal */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Legal</h3>
            <ul className="space-y-2 text-xs sm:text-sm">
              <li>
                <a
                  href="/privacy"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Política de Privacidade
                </a>
              </li>
              <li>
                <a
                  href="/disclaimer"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Termos e Condições
                </a>
              </li>
            </ul>
          </div>

          {/* Responsible Gambling */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Jogo Responsável</h3>
            <p className="text-xs leading-relaxed text-white/70">
              Jogue com responsabilidade e aposte apenas o que pode perder. Se o jogo se tornar um problema, procure ajuda de organizações de apoio locais.
            </p>
          </div>
        </div>

        {/* Responsible Gambling Organizations */}
        <div 
          className="mb-6 sm:mb-8 rounded-xl border p-3 sm:p-4 backdrop-blur-sm sm:rounded-2xl"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
            background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
            boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
          }}
        >
          <div className="flex flex-wrap items-center justify-center gap-2 sm:gap-3 md:gap-4">
            <a
              href="https://www.srij.turismodeportugal.pt"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center gap-1.5 sm:gap-2.5 rounded-lg border px-2 sm:px-4 py-1.5 sm:py-2.5 transition-all duration-200 hover:opacity-100"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
                backgroundColor: 'rgba(255, 255, 255, 0.05)',
              }}
              aria-label="SRIJ – Serviço de Regulação e Inspeção de Jogos"
            >
              <div 
                className="flex h-6 w-6 sm:h-8 sm:w-8 items-center justify-center rounded-full transition-colors"
                style={{
                  backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.2),
                }}
              >
                <svg
                  className="h-3 w-3 sm:h-4 sm:w-4"
                  style={{ color: colorTheme.buttonBadge }}
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth="2.5"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"
                  />
                </svg>
              </div>
              <span className="text-xs sm:text-sm font-medium text-white/90 transition-colors group-hover:text-white">
                SRIJ
              </span>
            </a>
            <a
              href="https://www.icad.pt"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center gap-1.5 sm:gap-2.5 rounded-lg border border-white/10 bg-white/5 px-2 sm:px-4 py-1.5 sm:py-2.5 transition-all duration-200 hover:border-pink-400/50 hover:bg-pink-500/10 hover:opacity-100 hover:shadow-lg hover:shadow-pink-500/20"
              aria-label="ICAD – Instituto para os Comportamentos Aditivos e as Dependências"
            >
              <div className="flex h-6 w-6 sm:h-8 sm:w-8 items-center justify-center rounded-full bg-pink-500/20 transition-colors group-hover:bg-pink-500/30">
                <svg
                  className="h-3 w-3 sm:h-4 sm:w-4 text-pink-400"
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth="2.5"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"
                  />
                </svg>
              </div>
              <span className="text-xs sm:text-sm font-medium text-white/90 transition-colors group-hover:text-white">
                ICAD
              </span>
            </a>
            <a
              href="tel:1414"
              className="group flex items-center gap-1.5 sm:gap-2.5 rounded-lg border border-white/10 bg-white/5 px-2 sm:px-4 py-1.5 sm:py-2.5 transition-all duration-200 hover:border-green-400/50 hover:bg-green-500/10 hover:opacity-100 hover:shadow-lg hover:shadow-green-500/20"
              aria-label="Linha Vida - 1414"
            >
              <div className="flex h-6 w-6 sm:h-8 sm:w-8 items-center justify-center rounded-full bg-green-500/20 transition-colors group-hover:bg-green-500/30">
                <svg
                  className="h-3 w-3 sm:h-4 sm:w-4 text-green-400"
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth="2.5"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z"
                  />
                </svg>
              </div>
              <span className="text-xs sm:text-sm font-medium text-white/90 transition-colors group-hover:text-white">
                Linha Vida - 1414
              </span>
            </a>
            <a
              href="https://www.jogadoresanonimos.pt"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center gap-1.5 sm:gap-2.5 rounded-lg border border-white/10 bg-white/5 px-2 sm:px-4 py-1.5 sm:py-2.5 transition-all duration-200 hover:border-purple-400/50 hover:bg-purple-500/10 hover:opacity-100 hover:shadow-lg hover:shadow-purple-500/20"
              aria-label="Jogadores Anónimos"
            >
              <div className="flex h-6 w-6 sm:h-8 sm:w-8 items-center justify-center rounded-full bg-purple-500/20 transition-colors group-hover:bg-purple-500/30">
                <svg
                  className="h-3 w-3 sm:h-4 sm:w-4 text-purple-400"
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth="2.5"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"
                  />
                </svg>
              </div>
              <span className="text-xs sm:text-sm font-medium text-white/90 transition-colors group-hover:text-white">
                Jogadores Anónimos
              </span>
            </a>
            <div className="flex items-center gap-1.5 sm:gap-2.5 rounded-lg bg-gradient-to-r from-orange-500 via-orange-600 to-red-600 px-3 sm:px-5 py-1.5 sm:py-2.5 font-bold text-white shadow-lg shadow-orange-500/30 transition-all duration-200 hover:scale-105 hover:shadow-xl hover:shadow-orange-500/40">
              <svg
                className="h-4 w-4 sm:h-5 sm:w-5"
                fill="none"
                viewBox="0 0 24 24"
                stroke="currentColor"
                strokeWidth="2.5"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"
                />
              </svg>
              <span className="text-xs sm:text-sm md:text-base">Jogue com Responsabilidade</span>
            </div>
          </div>
        </div>

        {/* Important Notice */}
        <div 
          className="mb-6 sm:mb-8 rounded-xl border p-4 sm:p-6 shadow-lg backdrop-blur-sm sm:rounded-2xl"
          style={{
            borderColor: hexToRgba(colorTheme.welcomeBonus, 0.25),
            background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
            boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.welcomeBonus, 0.1)}`,
          }}
        >
          <div className="flex flex-col sm:flex-row items-start gap-3 sm:gap-4">
            <div className="flex-shrink-0">
              <div 
                className="flex h-8 w-8 sm:h-10 sm:w-10 items-center justify-center rounded-full"
                style={{
                  background: `linear-gradient(to bottom right, ${colorTheme.welcomeBonus}, ${colorTheme.welcomeBonus})`,
                  boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.welcomeBonus, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                }}
              >
                <svg
                  className="h-5 w-5 sm:h-6 sm:w-6"
                  style={{ color: colorTheme.welcomeBonus }}
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth="2"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"
                  />
                </svg>
              </div>
            </div>
            <div className="flex-1">
              <h4 
                className="mb-2 text-sm font-bold text-white sm:text-base"
              >
                Aviso Importante
              </h4>
              <p 
                className="text-xs leading-relaxed text-white sm:text-sm"
              >
                Este site contém links de afiliados. Podemos receber uma comissão se se
                registar através dos nossos links. Por favor, aposte com responsabilidade e
                aposte apenas o que pode perder. O jogo pode ser viciante.
              </p>
            </div>
          </div>
        </div>

        {/* Data Collection Notice */}
        <div className="border-t border-white/10 pt-4 sm:pt-5">
          <p className="text-xs leading-relaxed text-white/50 text-center">
            Utilizamos rastreamento de visitantes e dados de cliques para melhorar o nosso website e serviços.{" "}
            <a
              href="/privacy"
              className="text-white/70 underline decoration-white/30 underline-offset-2 transition-colors hover:text-white/90 hover:decoration-white/50"
            >
              Saiba mais
            </a>
            .
          </p>
        </div>

        {/* Copyright */}
        <div className="border-t border-white/10 pt-4 sm:pt-6 text-center text-xs text-white/60">
          <p>
            © {new Date().getFullYear()} {siteInfo.siteName}. Todos os direitos reservados.
          </p>
        </div>
      </div>
    </footer>
  );
}
