/**
 * Language system types and constants
 * Supports: en (English), pt (Portuguese), dk (Danish), fr (French), nl (Dutch), sv (Swedish), es (Spanish)
 */

export type LanguageCode = "en" | "pt" | "dk" | "fr" | "nl" | "sv" | "es";

export const SUPPORTED_LANGUAGES: readonly LanguageCode[] = ["en", "pt", "dk", "fr", "nl", "sv", "es"] as const;

export const DEFAULT_LANGUAGE: LanguageCode = "pt";

export const LANGUAGE_NAMES: Record<LanguageCode, string> = {
  en: "English",
  pt: "Português",
  dk: "Dansk",
  fr: "Français",
  nl: "Nederlands",
  sv: "Svenska",
  es: "Español",
} as const;

/**
 * Validates if a string is a supported language code
 */
export function isValidLanguage(code: string | null | undefined): code is LanguageCode {
  return SUPPORTED_LANGUAGES.includes(code as LanguageCode);
}

/**
 * Gets language code from string, falls back to default
 */
export function getLanguageCode(code: string | null | undefined): LanguageCode {
  return isValidLanguage(code) ? code : DEFAULT_LANGUAGE;
}

