# Multi-Language System Implementation

## Overview

This document describes the comprehensive multi-language system implemented for the Next.js betting affiliate website. The system supports **English (en)**, **Portuguese (pt)**, and **Danish (dk)** with full SEO, Google Ads compliance, and language-specific compliance content.

## Architecture

### Core Principles

1. **Single Dynamic Architecture**: No URL prefixes, no duplicated pages. Language is controlled via cookies.
2. **Admin Panel**: Always in English, with a Language Switch tab to control the public site.
3. **Data Directory**: Remains unchanged - all language logic is dynamic.
4. **Cookie-Based Persistence**: Language preference stored in cookies (SSR-safe).

## Language System Components

### 1. Language Types & Constants (`src/types/language.ts`)

- Defines supported languages: `en`, `pt`, `dk`
- Default language: `pt`
- Validation utilities for language codes

### 2. Language Management (`src/lib/language.ts`)

- `getLanguage()`: Server-side function to read language from cookies (cached per request)
- `setLanguage()`: Server-side function to set language cookie
- `getLanguageFromRequest()`: Utility for middleware/headers

### 3. Translations System (`src/lib/translations.ts`)

Centralized translation object with all UI text:
- Hero section (badge, tagline)
- Features section (title, subtitle, all 4 features)
- Partners section (heading, buttons, terms)
- Footer (all labels)

**Adding a new language:**
1. Add language code to `SUPPORTED_LANGUAGES` in `src/types/language.ts`
2. Add translations object in `src/lib/translations.ts`
3. Create footer component in `src/components/footer/Footer{CODE}.tsx`
4. Update footer switch in `src/components/footer/Footer.tsx`

## SEO & Indexing

### Metadata Generation (`src/lib/metadata.ts`)

**Language-Specific Metadata:**
- `<html lang="">` attribute updates per language
- `<title>` and `<meta name="description">` use language-specific translations
- OpenGraph `locale` changes per language (`en_GB`, `pt_PT`, `da_DK`)
- Twitter Card metadata includes language-specific content

**Hreflang Tags:**
- All supported languages are included in `alternates.languages`
- Each language points to the same canonical URL (single dynamic architecture)
- Prevents duplicate content penalties
- Ensures Google indexes each language correctly

**Example:**
```html
<link rel="alternate" hreflang="en" href="https://www.betranked.com/" />
<link rel="alternate" hreflang="pt" href="https://www.betranked.com/" />
<link rel="alternate" hreflang="dk" href="https://www.betranked.com/" />
<link rel="canonical" href="https://www.betranked.com/" />
```

### Google Ads Compliance

**Landing Page Language Matching:**
- When language changes, ALL metadata updates automatically
- Title, description, and OpenGraph tags match the selected language
- Ensures ad landing page language always matches the ad language
- No mixed-language content that could violate Google Ads policies

**Responsible Gambling Messaging:**
- Footer compliance blocks are language-specific (not just translated text)
- Each language has its own HTML structure and images
- Images loaded from `/public/footer-{lang}/` folders
- Affiliate disclosure text matches local compliance expectations

## Footer Architecture

### Language-Specific Components

Each language has its own footer component with complete markup:

- **`FooterEN.tsx`**: UK responsible gambling organizations, UK compliance wording
- **`FooterPT.tsx`**: Portuguese organizations (SRIJ, ICAD, Linha Vida), Portuguese compliance
- **`FooterDK.tsx`**: Danish organizations (StopSpillet, ROFUS), Danish compliance

**Key Features:**
- Different HTML structures per language (not conditional text)
- Images in `/public/footer-{lang}/` folders
- Language-specific compliance organizations and links
- Important Notice section with localized legal wording

### Image Paths

- English: `/footer-uk/gambleaware.webp`, `/footer-uk/gamstop.svg`, etc.
- Portuguese: Uses SVG icons (no separate folder needed for current implementation)
- Danish: `/footer-dk/stopspillet.jpeg`, `/footer-dk/rofus.jpeg`, etc.

## Admin Panel

### Language Switch Tab (`src/components/admin/LanguageTab.tsx`)

**Features:**
- Visual language selector with current language highlighted
- One-click language switching
- Automatic page reload after language change
- Success/error messaging
- Admin panel UI remains in English

**API Endpoint:** `/api/language`
- `POST`: Set language (sets cookie)
- `GET`: Get current language (reads cookie)

## Component Updates

### Server Components (Language-Aware)

1. **`src/app/layout.tsx`**:
   - Reads language from cookies
   - Sets `<html lang="">` attribute
   - Passes language to Footer component

2. **`src/app/page.tsx`**:
   - Reads language and gets translations
   - Passes translations to HeroSection and FeaturesSection
   - Generates language-specific metadata

3. **`src/components/HeroSection.tsx`**:
   - Accepts `translations` prop
   - Uses `translations.hero.badge` and `translations.hero.tagline`

4. **`src/components/FeaturesSection.tsx`**:
   - Accepts `translations` prop
   - Uses `translations.features.*` for all content

5. **`src/components/footer/Footer.tsx`**:
   - Accepts `language` prop
   - Renders appropriate language-specific footer component

## How Language Affects SEO & Indexing

### 1. HTML Lang Attribute
```html
<!-- English -->
<html lang="en">

<!-- Portuguese -->
<html lang="pt">

<!-- Danish -->
<html lang="da">
```

### 2. Metadata Updates
- **Title**: Language-specific browser title
- **Description**: Language-specific meta description
- **OpenGraph Locale**: `en_GB`, `pt_PT`, `da_DK`
- **Hreflang Tags**: All languages included for proper indexing

### 3. Content Language
- All UI text changes per language
- Footer compliance blocks are language-specific
- Structured data (if added) would use language-specific content

### 4. Google Ads Compliance
- Landing page language always matches ad language
- No mixed-language content
- Compliance messaging is localized and visible
- Affiliate disclosure matches local expectations

## How to Add a New Language

### Step 1: Add Language Code
```typescript
// src/types/language.ts
export type LanguageCode = "en" | "pt" | "dk" | "fr"; // Add "fr"
export const SUPPORTED_LANGUAGES: readonly LanguageCode[] = ["en", "pt", "dk", "fr"];
export const LANGUAGE_NAMES: Record<LanguageCode, string> = {
  // ... existing
  fr: "Français",
};
```

### Step 2: Add Translations
```typescript
// src/lib/translations.ts
const translations: Record<LanguageCode, Translations> = {
  // ... existing
  fr: {
    hero: { badge: "...", tagline: "..." },
    features: { /* ... */ },
    // ... all translations
  },
};
```

### Step 3: Create Footer Component
```typescript
// src/components/footer/FooterFR.tsx
export function FooterFR({ siteInfo }: FooterFRProps) {
  // French-specific footer with French compliance organizations
}
```

### Step 4: Update Footer Switch
```typescript
// src/components/footer/Footer.tsx
switch (language) {
  // ... existing cases
  case "fr":
    return <FooterFR siteInfo={siteInfo} />;
}
```

### Step 5: Update Metadata Locale
```typescript
// src/lib/metadata.ts
const localeMap: Record<LanguageCode, string> = {
  // ... existing
  fr: "fr_FR",
};
```

### Step 6: Add Page Descriptions
```typescript
// src/app/page.tsx
const descriptionsByLang = {
  // ... existing
  fr: "Description en français...",
};
```

## File Structure

```
src/
├── types/
│   └── language.ts              # Language types & constants
├── lib/
│   ├── language.ts               # Language cookie management
│   ├── translations.ts           # All UI translations
│   └── metadata.ts               # Language-aware metadata
├── components/
│   ├── footer/
│   │   ├── Footer.tsx            # Footer router
│   │   ├── FooterEN.tsx          # English footer
│   │   ├── FooterPT.tsx          # Portuguese footer
│   │   └── FooterDK.tsx          # Danish footer
│   ├── HeroSection.tsx            # Uses translations
│   └── FeaturesSection.tsx        # Uses translations
├── app/
│   ├── layout.tsx                 # Sets html lang attribute
│   ├── page.tsx                   # Language-aware homepage
│   └── api/
│       └── language/
│           └── route.ts           # Language API endpoint
└── components/admin/
    └── LanguageTab.tsx            # Admin language switcher

public/
├── footer-uk/                     # English footer images
│   ├── gambleaware.webp
│   ├── gamstop.svg
│   └── ...
├── footer-pt/                     # Portuguese footer images (if needed)
└── footer-dk/                     # Danish footer images
    ├── stopspillet.jpeg
    ├── rofus.jpeg
    └── ...
```

## Testing Checklist

- [ ] Language switch in admin panel works
- [ ] Page reloads after language change
- [ ] HTML lang attribute updates correctly
- [ ] Metadata (title, description) updates per language
- [ ] Footer content changes per language
- [ ] Footer images load from correct folders
- [ ] Hreflang tags are present in HTML
- [ ] OpenGraph locale updates per language
- [ ] All UI text translates correctly
- [ ] Admin panel remains in English
- [ ] Cookie persists across page loads
- [ ] Google Ads landing page language matches ad language

## SEO Best Practices Applied

1. ✅ **Hreflang Tags**: All languages included for proper indexing
2. ✅ **Canonical URLs**: Stable, language-agnostic canonical URLs
3. ✅ **HTML Lang Attribute**: Correct per language
4. ✅ **Metadata Localization**: Title, description, OpenGraph per language
5. ✅ **No Duplicate Content**: Single URL architecture with language context
6. ✅ **Google Ads Compliance**: Landing page language matches ad language
7. ✅ **Responsible Gambling**: Language-specific compliance messaging

## Google Ads Compliance

### Landing Page Language Matching
- When a user clicks a Google Ad, the landing page language is determined by the cookie
- Admin can set the language to match the ad campaign language
- Metadata (title, description) updates automatically
- Footer compliance content matches the ad language

### Affiliate Disclosure
- Each language has its own Important Notice section
- Wording matches local compliance expectations
- Visible and properly formatted per language

### Responsible Gambling
- Language-specific organizations and links
- Local helpline numbers and resources
- Country-specific regulatory information

## Future Enhancements

1. **Auto-Detection**: Detect language from Accept-Language header on first visit
2. **Language Switcher UI**: Add public-facing language switcher (optional)
3. **More Languages**: Easy to add following the 6-step process above
4. **Structured Data**: Add language-specific structured data (Schema.org)
5. **Analytics**: Track language preferences for insights

## Summary

The multi-language system is:
- ✅ **Scalable**: Easy to add new languages
- ✅ **SEO-Safe**: Proper hreflang, canonical URLs, metadata
- ✅ **Google Ads Compliant**: Landing page language matches ad language
- ✅ **Maintainable**: Centralized translations, component-per-language footers
- ✅ **Performance-Optimized**: Server-side rendering, cookie-based persistence
- ✅ **Admin-Friendly**: Simple language switch in admin panel

All requirements met:
- ✅ Data directory unchanged
- ✅ Admin panel in English only
- ✅ Language switch tab in admin
- ✅ Single dynamic architecture (no URL prefixes)
- ✅ Language-specific footers with compliance content
- ✅ Full SEO metadata and hreflang support
- ✅ Google Ads compliance maintained

