import type { SiteInfo } from "@/types/data";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface FooterENProps {
  readonly siteInfo: SiteInfo;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

export function FooterEN({ siteInfo, language, theme }: FooterENProps) {
  const colorTheme = getColorTheme(theme);
  
  return (
    <footer 
      className="relative border-t"
      style={{
        borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
        background: `linear-gradient(to bottom right, ${colorTheme.cardBackground}, ${hexToRgba(colorTheme.cardBackground, 0.95)}, ${colorTheme.cardBackground})`,
      }}
    >
      {/* Subtle background gradients - theme-based */}
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 20% 30%, ${hexToRgba(colorTheme.buttonBadge, 0.08)}, transparent 50%)`,
        }}
      />
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 80% 70%, ${hexToRgba(colorTheme.welcomeBonus, 0.06)}, transparent 50%)`,
        }}
      />
      
      {/* Animated grid pattern */}
      <div className="absolute inset-0 bg-[linear-gradient(rgba(255,255,255,0.02)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,0.02)_1px,transparent_1px)] bg-[size:30px_30px] opacity-40" />
      
      <div className="relative mx-auto max-w-6xl px-4 py-6 sm:px-6 sm:py-12 lg:py-16">
        <div className="grid gap-6 sm:gap-8 md:grid-cols-2 lg:grid-cols-4 mb-8 sm:mb-12">
          {/* Contact */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Contact</h3>
            <div className="space-y-2 sm:space-y-3 text-xs sm:text-sm">
              <p>
                <span className="font-semibold text-white/80">Email:</span>{" "}
                <a
                  href={`mailto:${siteInfo.email}`}
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  {siteInfo.email}
                </a>
              </p>
              <p>
                <span className="font-semibold text-white/80">Phone:</span>{" "}
                <a
                  href={`tel:${siteInfo.phone.replace(/\s/g, "")}`}
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  {siteInfo.phone}
                </a>
              </p>
              <p>
                <span className="font-semibold text-white/80">Address:</span>{" "}
                <span className="text-white/70">{siteInfo.address}</span>
              </p>
            </div>
          </div>

          {/* Quick Links */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Quick Links</h3>
            <ul className="space-y-2 text-xs sm:text-sm">
              <li>
                <a
                  href="/"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Home
                </a>
              </li>
              <li>
                <a
                  href="/about"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  About Us
                </a>
              </li>
              <li>
                <a
                  href="/contact"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Contact
                </a>
              </li>
            </ul>
          </div>

          {/* Legal */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Legal</h3>
            <ul className="space-y-2 text-xs sm:text-sm">
              <li>
                <a
                  href="/privacy"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Privacy Policy
                </a>
              </li>
              <li>
                <a
                  href="/disclaimer"
                  className="text-white/70 transition-colors hover:text-white hover:underline"
                >
                  Terms & Conditions
                </a>
              </li>
            </ul>
          </div>

          {/* Responsible Gambling */}
          <div className="space-y-3 sm:space-y-4">
            <h3 className="text-base sm:text-lg font-bold text-white">Responsible Gambling</h3>
            <p className="text-xs leading-relaxed text-white/70">
              Play responsibly and only wager what you can afford to lose. If gambling becomes a problem, seek help from local support organizations.
            </p>
          </div>
        </div>

        {/* Responsible Gambling Organizations */}
        <div 
          className="mb-6 sm:mb-8 rounded-xl p-3 sm:p-4 backdrop-blur-sm sm:rounded-2xl"
          style={{
            border: `1px solid ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
            background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0.05)}, ${hexToRgba(colorTheme.buttonBadge, 0.03)}, transparent)`,
            boxShadow: `0 10px 15px -3px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
          }}
        >
          <div className="flex flex-wrap items-center justify-center gap-2 sm:gap-3 md:gap-4">
            <a
              href="https://www.begambleaware.org/"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center justify-center rounded-lg border bg-white/5 p-1.5 sm:p-2 transition-all duration-300 hover:scale-105"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              }}
              title="BeGambleAware"
            >
              <img
                src="/footer-uk/gambleaware.webp"
                alt="BeGambleAware"
                className="h-5 w-auto sm:h-7 object-contain opacity-80 transition-opacity duration-300 group-hover:opacity-100"
                loading="lazy"
              />
            </a>
            <a
              href="https://www.raig.org/"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center justify-center rounded-lg border bg-white/5 p-1.5 sm:p-2 transition-all duration-300 hover:scale-105"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              }}
              title="RAiG"
            >
              <img
                src="/footer-uk/Raig.svg"
                alt="RAiG"
                className="h-5 w-auto sm:h-7 object-contain opacity-80 transition-opacity duration-300 group-hover:opacity-100"
                loading="lazy"
              />
            </a>
            <a
              href="https://www.gamstop.co.uk/"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center justify-center rounded-lg border bg-white/5 p-1.5 sm:p-2 transition-all duration-300 hover:scale-105"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              }}
              title="GAMSTOP"
            >
              <img
                src="/footer-uk/gamstop.svg"
                alt="GAMSTOP"
                className="h-5 w-auto sm:h-7 object-contain opacity-80 transition-opacity duration-300 group-hover:opacity-100"
                loading="lazy"
              />
            </a>
            <a
              href="https://www.gamcare.org.uk/"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center justify-center rounded-lg border bg-white/5 p-1.5 sm:p-2 transition-all duration-300 hover:scale-105"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              }}
              title="GamCare"
            >
              <img
                src="/footer-uk/gamecare.svg"
                alt="GamCare"
                className="h-5 w-auto sm:h-7 object-contain opacity-80 transition-opacity duration-300 group-hover:opacity-100"
                loading="lazy"
              />
            </a>
            <a
              href="https://www.gamblingtherapy.org/"
              target="_blank"
              rel="noopener noreferrer"
              className="group flex items-center justify-center rounded-lg border bg-white/5 p-1.5 sm:p-2 transition-all duration-300 hover:scale-105"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              }}
              title="Gambling Therapy"
            >
              <img
                src="/footer-uk/gamblingtherapy.svg"
                alt="Gambling Therapy"
                className="h-5 w-auto sm:h-7 object-contain opacity-80 transition-opacity duration-300 group-hover:opacity-100"
                loading="lazy"
              />
            </a>
            <div 
              className="flex items-center justify-center rounded-lg border bg-white/5 p-1.5 sm:p-2"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              }}
            >
              <img
                src="/footer-uk/18+.svg"
                alt="18+ Only"
                className="h-5 w-auto sm:h-7 object-contain"
                loading="lazy"
              />
            </div>
          </div>
        </div>

        {/* Important Notice */}
        <div 
          className="mb-6 sm:mb-8 rounded-xl border-2 p-4 sm:p-6 shadow-lg backdrop-blur-sm sm:rounded-2xl"
          style={{
            borderColor: hexToRgba(colorTheme.welcomeBonus, 0.4),
            background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.welcomeBonus, 0.1)}, ${hexToRgba(colorTheme.welcomeBonus, 0.05)}, transparent)`,
            boxShadow: `0 10px 15px -3px ${hexToRgba(colorTheme.welcomeBonus, 0.1)}`,
          }}
        >
          <div className="flex flex-col sm:flex-row items-start gap-3 sm:gap-4">
            <div className="flex-shrink-0">
              <div 
                className="flex h-8 w-8 sm:h-10 sm:w-10 items-center justify-center rounded-full border"
                style={{
                  backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.2),
                  borderColor: hexToRgba(colorTheme.welcomeBonus, 0.3),
                }}
              >
                <svg
                  className="h-5 w-5 sm:h-6 sm:w-6"
                  style={{ color: colorTheme.welcomeBonus }}
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth="2"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"
                  />
                </svg>
              </div>
            </div>
            <div className="flex-1">
              <h4 
                className="mb-2 text-sm font-bold text-white sm:text-base"
              >
                Important Notice
              </h4>
              <p 
                className="text-xs leading-relaxed text-white sm:text-sm"
              >
                This site contains affiliate links. We may receive a commission if you sign up
                through our links. Please gamble responsibly and only bet what you can afford to
                lose. Gambling can be addictive. If you need help, contact{" "}
                <a
                  href="https://www.begambleaware.org"
                  target="_blank"
                  rel="noopener noreferrer"
                  className="font-semibold text-white underline underline-offset-2 transition-colors hover:opacity-80"
                >
                  BeGambleAware
                </a>{" "}
                or{" "}
                <a
                  href="https://www.gamcare.org.uk"
                  target="_blank"
                  rel="noopener noreferrer"
                  className="font-semibold text-white underline underline-offset-2 transition-colors hover:opacity-80"
                >
                  GamCare
                </a>
                .
              </p>
            </div>
          </div>
        </div>

        {/* Data Collection Notice */}
        <div className="border-t border-white/10 pt-4 sm:pt-5">
          <p className="text-xs leading-relaxed text-white/50 text-center">
            We use visitor tracking and click data to improve our website and services.{" "}
            <a
              href="/privacy"
              className="text-white/70 underline decoration-white/30 underline-offset-2 transition-colors hover:text-white/90 hover:decoration-white/50"
            >
              Learn more
            </a>
            .
          </p>
        </div>

        {/* Copyright */}
        <div className="border-t border-white/10 pt-4 sm:pt-6 text-center text-xs text-white/60">
          <p>
            © {new Date().getFullYear()} {siteInfo.siteName}. All rights reserved.
          </p>
        </div>
      </div>
    </footer>
  );
}
