# Runtime JSON Data Refactoring

## Summary
Refactored the application to read JSON files at **runtime** instead of bundling them at build time. This ensures that JSON data updates in cPanel are immediately reflected without requiring a rebuild.

---

## Changes Made

### 1. Created Runtime Data Utility (`src/lib/data.ts`)
- **`getPartners()`**: Reads `data/partners.json` at runtime using `fs.readFileSync`
- **`getSiteInfo()`**: Reads `data/siteinfo.json` at runtime using `fs.readFileSync`
- **`ensureDataDir()`**: Automatically creates `data/` directory if it doesn't exist
- All functions handle errors gracefully with fallback values

### 2. Updated All Pages to Use Runtime Data

#### **`src/app/page.tsx`**
- ❌ Removed: `import partnersData from "../../data/partners.json"`
- ❌ Removed: `import siteInfo from "../../data/siteinfo.json"`
- ✅ Added: `import { getPartners, getSiteInfo } from "@/lib/data"`
- ✅ Changed: `generateMetadata()` to async function
- ✅ Changed: Data read at runtime in component: `const partners = getPartners()`

#### **`src/app/layout.tsx`**
- ❌ Removed: `import siteInfo from "../../data/siteinfo.json"`
- ✅ Added: `import { getSiteInfo } from "@/lib/data"`
- ✅ Changed: `metadata` export to `generateMetadata()` async function
- ✅ Changed: `siteInfo` read at runtime in component

#### **`src/app/about/page.tsx`**
- ❌ Removed: `import siteInfo from "../../../data/siteinfo.json"`
- ✅ Added: `import { getSiteInfo } from "@/lib/data"`
- ✅ Changed: `generateMetadata()` to async function
- ✅ Changed: `siteInfo` read at runtime in component

#### **`src/app/contact/page.tsx`**
- ❌ Removed: `import siteInfo from "../../../data/siteinfo.json"`
- ✅ Added: `import { getSiteInfo } from "@/lib/data"`
- ✅ Changed: `generateMetadata()` to async function
- ✅ Changed: `siteInfo` read at runtime in component

#### **`src/lib/metadata.ts`**
- ❌ Removed: `import siteInfo from "../../data/siteinfo.json"`
- ✅ Added: `import { getSiteInfo } from "./data"`
- ✅ Changed: `createMetadata()` now calls `getSiteInfo()` at runtime

### 3. Updated API Routes

All API routes now ensure the `data/` directory exists before reading/writing:

#### **`src/app/api/admin/partners/route.ts`**
- ✅ Added `ensureDataDir()` function
- ✅ Checks if file exists before reading
- ✅ Creates directory if missing

#### **`src/app/api/admin/siteinfo/route.ts`**
- ✅ Added `ensureDataDir()` function
- ✅ Checks if file exists before reading/writing
- ✅ Creates directory if missing

#### **`src/app/api/admin/partners/reorder/route.ts`**
- ✅ Added `ensureDataDir()` function
- ✅ Ensures directory exists before writing

---

## How It Works

### Before (Build-time Bundling)
```typescript
// ❌ This bundles JSON into the build
import siteInfo from "../../data/siteinfo.json";

export default function Page() {
  return <div>{siteInfo.siteName}</div>; // Static, can't change
}
```

### After (Runtime Reading)
```typescript
// ✅ This reads JSON at runtime
import { getSiteInfo } from "@/lib/data";

export default function Page() {
  const siteInfo = getSiteInfo(); // Read fresh on each request
  return <div>{siteInfo.siteName}</div>; // Dynamic, updates immediately
}
```

---

## Benefits

1. **✅ Immediate Updates**: Changes to JSON files in cPanel are reflected immediately
2. **✅ No Rebuild Required**: Admin panel changes work without rebuilding
3. **✅ cPanel Compatible**: Works with read-only build folders
4. **✅ Error Handling**: Graceful fallbacks if files don't exist
5. **✅ Auto-Create Directories**: `data/` directory is created automatically

---

## File Structure

```
/
├── data/                    # Runtime data (not bundled)
│   ├── partners.json        # Read/written at runtime
│   └── siteinfo.json        # Read/written at runtime
├── src/
│   ├── lib/
│   │   └── data.ts          # Runtime data utilities
│   └── app/
│       ├── page.tsx         # Uses getPartners(), getSiteInfo()
│       ├── layout.tsx        # Uses getSiteInfo()
│       └── api/
│           └── admin/        # All routes ensure data/ exists
```

---

## Testing

### Verify Runtime Reading
1. Start the application: `npm start`
2. Make a change in admin panel (e.g., update site name)
3. Refresh the page - changes should appear **immediately**
4. No rebuild required!

### Verify Directory Creation
1. Delete `data/` directory
2. Start the application
3. `data/` directory should be created automatically
4. Application should work normally

---

## Important Notes

1. **No Static Imports**: Never use `import ... from "data/...json"` - always use `getPartners()` or `getSiteInfo()`
2. **Server-Side Only**: These functions use `fs.readFileSync` which only works server-side
3. **Error Handling**: Functions return default values if files don't exist
4. **Performance**: Reading JSON files is fast, but happens on every request (not cached)

---

## Migration Checklist

- [x] Created `src/lib/data.ts` utility
- [x] Removed all static JSON imports
- [x] Updated all pages to use runtime data
- [x] Updated API routes to ensure data directory exists
- [x] Changed `generateMetadata()` to async where needed
- [x] Build passes successfully
- [x] All functionality preserved

---

**Status**: ✅ Complete - All JSON files now read at runtime

