import type { Metadata } from "next";
import { createMetadata } from "@/lib/metadata";
import { getLanguage } from "@/lib/language";
import { getTheme } from "@/lib/theme";
import { getTranslations } from "@/lib/translations";
import { getSiteInfo } from "@/lib/data";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";
import { FormattedDate } from "@/components/FormattedDate";

export async function generateMetadata(): Promise<Metadata> {
  const language = await getLanguage();
  const translations = getTranslations(language);
  
  return createMetadata({
    title: translations.privacy.title,
    description: translations.privacy.subtitle,
    path: "/privacy",
    language,
  });
}

export default async function PrivacyPage() {
  const language = await getLanguage();
  const theme = await getTheme();
  const translations = getTranslations(language);
  const siteInfo = getSiteInfo();
  const colorTheme = getColorTheme(theme);
  
  // Create a stable date string for "last updated" to prevent hydration mismatches
  // Use UTC to ensure consistency across server and client
  const today = new Date();
  const lastUpdatedDate = `${today.getUTCFullYear()}-${String(today.getUTCMonth() + 1).padStart(2, '0')}-${String(today.getUTCDate()).padStart(2, '0')}`;
  
  // Helper function to replace {siteName} placeholder
  const replaceSiteName = (text: string): string => {
    return text.replace(/{siteName}/g, siteInfo.siteName);
  };

  return (
    <div className="space-y-16">
      <section 
        className="relative overflow-hidden rounded-2xl border px-8 py-10 text-white shadow-2xl sm:rounded-3xl sm:px-12 sm:py-12"
        style={{
          borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
          boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
        }}
      >
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
          }}
        />
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
          }}
        />
        <div className="relative space-y-6 text-center">
          <span 
            className="inline-flex items-center gap-2 rounded-full border px-4 py-2 text-xs font-semibold uppercase tracking-[0.15em] backdrop-blur-sm"
            style={{
              borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              color: hexToRgba(colorTheme.buttonBadge, 0.9),
            }}
          >
            {translations.privacy.badge}
          </span>
          <h1 className="text-4xl font-bold leading-[1.1] tracking-tight text-white sm:text-6xl lg:text-7xl">
            {translations.privacy.title}
          </h1>
          <p className="mx-auto max-w-3xl text-lg leading-relaxed text-white/90 sm:text-xl">
            {translations.privacy.subtitle}
          </p>
        </div>
      </section>

      <section 
        className="relative space-y-8 overflow-hidden rounded-2xl border p-8 shadow-xl backdrop-blur-sm sm:rounded-3xl sm:p-12"
        style={{
          borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
          boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
        }}
      >
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
          }}
        />
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
          }}
        />
        <div className="relative prose prose-slate max-w-none space-y-6 text-white">
          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.informationWeCollect.title}
            </h2>
            <p className="leading-relaxed text-white">
              {replaceSiteName(translations.privacy.sections.informationWeCollect.description)}
            </p>
            <ul className="list-disc space-y-2 pl-6 text-white">
              {translations.privacy.sections.informationWeCollect.items.map((item, index) => (
                <li key={index}>{item}</li>
              ))}
            </ul>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.howWeUse.title}
            </h2>
            <p className="leading-relaxed text-white">
              {translations.privacy.sections.howWeUse.description}
            </p>
            <ul className="list-disc space-y-2 pl-6 text-white">
              {translations.privacy.sections.howWeUse.items.map((item, index) => (
                <li key={index}>{item}</li>
              ))}
            </ul>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.dataProtection.title}
            </h2>
            <p className="leading-relaxed text-white">
              {replaceSiteName(translations.privacy.sections.dataProtection.description)}
            </p>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.cookies.title}
            </h2>
            <p className="leading-relaxed text-white">
              {replaceSiteName(translations.privacy.sections.cookies.description)}
            </p>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.thirdParty.title}
            </h2>
            <p className="leading-relaxed text-white">
              {translations.privacy.sections.thirdParty.description}
            </p>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.yourRights.title}
            </h2>
            <p className="leading-relaxed text-white">
              {translations.privacy.sections.yourRights.description}
            </p>
            <ul className="list-disc space-y-2 pl-6 text-white">
              {translations.privacy.sections.yourRights.items.map((item, index) => (
                <li key={index}>{item}</li>
              ))}
            </ul>
            <p className="leading-relaxed text-white">
              {translations.privacy.sections.yourRights.contactText}{" "}
              <a href={`mailto:${siteInfo.email}`} className="text-white font-semibold hover:underline">
                {siteInfo.email}
              </a>
            </p>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.changes.title}
            </h2>
            <p className="leading-relaxed text-white">
              {translations.privacy.sections.changes.description}
            </p>
          </div>

          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-white">
              {translations.privacy.sections.contact.title}
            </h2>
            <p className="leading-relaxed text-white">
              {translations.privacy.sections.contact.description}{" "}
              <a href={`mailto:${siteInfo.email}`} className="text-white font-semibold hover:underline">
                {siteInfo.email}
              </a>
            </p>
            <p className="text-sm text-white/80">
              {translations.privacy.sections.contact.lastUpdated}{" "}
              <FormattedDate 
                date={lastUpdatedDate}
                locale={language === "en" ? "en-GB" : language === "pt" ? "pt-PT" : language === "dk" ? "da-DK" : language === "fr" ? "fr-FR" : language === "nl" ? "nl-NL" : language === "sv" ? "sv-SE" : "es-AR"}
                options={{ year: "numeric", month: "long", day: "numeric" }}
              />
            </p>
          </div>
        </div>
      </section>
    </div>
  );
}
