"use client";

import { useState, useEffect } from "react";
import { PartnersTab } from "./admin/PartnersTab";
import { ReorderTab } from "./admin/ReorderTab";
import { SiteInfoTab } from "./admin/SiteInfoTab";
import { ClicksTab } from "./admin/ClicksTab";
import { VisitorsTab } from "./admin/VisitorsTab";
import { LanguageTab } from "./admin/LanguageTab";
import { ThemeTab } from "./admin/ThemeTab";
import type { Partner } from "@/types/data";
import type { SiteInfo } from "@/types/data";

interface AdminDashboardProps {
  readonly onLogout: () => void;
}

type Tab = "partners" | "reorder" | "siteinfo" | "clicks" | "visitors" | "language" | "theme";

export function AdminDashboard({ onLogout }: AdminDashboardProps) {
  const [activeTab, setActiveTab] = useState<Tab>("partners");
  const [partners, setPartners] = useState<Partner[]>([]);
  const [siteInfo, setSiteInfo] = useState<SiteInfo | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [partnersRes, siteInfoRes] = await Promise.all([
        fetch("/api/admin/partners"),
        fetch("/api/admin/siteinfo"),
      ]);

      if (partnersRes.ok) {
        const partnersData = await partnersRes.json();
        setPartners(partnersData);
      }

      if (siteInfoRes.ok) {
        const siteInfoData = await siteInfoRes.json();
        setSiteInfo(siteInfoData);
      }
    } catch (error) {
      console.error("Failed to load data:", error);
    } finally {
      setLoading(false);
    }
  };

  const tabs: { id: Tab; label: string }[] = [
    { id: "partners", label: "Partners" },
    { id: "reorder", label: "Reorder" },
    { id: "siteinfo", label: "Site Info" },
    { id: "clicks", label: "Click Tracking" },
    { id: "visitors", label: "Visitors" },
    { id: "language", label: "Language Switch" },
    { id: "theme", label: "Theme Switch" },
  ];

  if (loading) {
    return (
      <div className="flex min-h-screen items-center justify-center">
        <div className="text-lg">Loading...</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-slate-50">
      <div className="border-b border-slate-200 bg-white shadow-sm">
        <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
          <div className="flex h-16 items-center justify-between">
            <h1 className="text-2xl font-bold text-slate-900">Admin Panel</h1>
            <button
              onClick={onLogout}
              className="rounded-lg bg-red-600 px-4 py-2 text-sm font-semibold text-white transition-colors hover:bg-red-700"
            >
              Logout
            </button>
          </div>
          <div className="flex space-x-1 border-t border-slate-200">
            {tabs.map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`px-6 py-3 text-sm font-medium transition-colors ${
                  activeTab === tab.id
                    ? "border-b-2 border-blue-600 text-blue-600"
                    : "text-slate-600 hover:text-slate-900"
                }`}
              >
                {tab.label}
              </button>
            ))}
          </div>
        </div>
      </div>

      <div className="mx-auto max-w-7xl px-4 py-8 sm:px-6 lg:px-8">
        {activeTab === "partners" && (
          <PartnersTab partners={partners} onUpdate={loadData} />
        )}
        {activeTab === "reorder" && (
          <ReorderTab partners={partners} onUpdate={loadData} />
        )}
        {activeTab === "siteinfo" && (
          <SiteInfoTab siteInfo={siteInfo} onUpdate={loadData} />
        )}
        {activeTab === "clicks" && (
          <ClicksTab partners={partners} />
        )}
        {activeTab === "visitors" && (
          <VisitorsTab />
        )}
        {activeTab === "language" && (
          <LanguageTab onUpdate={loadData} />
        )}
        {activeTab === "theme" && (
          <ThemeTab onUpdate={loadData} />
        )}
      </div>
    </div>
  );
}

