import type { Translations } from "@/lib/translations";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import type { SiteInfo } from "@/types/data";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface FeaturesSectionProps {
  readonly translations: Translations;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
  readonly siteInfo: SiteInfo;
}

const FEATURES_CONFIG = [
  {
    icon: (
      <path strokeLinecap="round" strokeLinejoin="round" d="M13 10V3L4 14h7v7l9-11h-7z" />
    ),
    key: "ultraRapido" as const,
    gradient: "from-blue-500 to-blue-600",
    shadow: "shadow-blue-500/30",
  },
  {
    icon: (
      <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
    ),
    key: "conforme" as const,
    gradient: "from-blue-500 to-blue-600",
    shadow: "shadow-blue-500/30",
  },
  {
    icon: (
      <path strokeLinecap="round" strokeLinejoin="round" d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z" />
    ),
    key: "mobileFirst" as const,
    gradient: "from-purple-500 to-purple-600",
    shadow: "shadow-purple-500/30",
  },
  {
    icon: (
      <>
        <path strokeLinecap="round" strokeLinejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
        <path strokeLinecap="round" strokeLinejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
      </>
    ),
    key: "transparente" as const,
    gradient: "from-yellow-500 to-yellow-600",
    shadow: "shadow-yellow-500/30",
  },
] as const;

export function FeaturesSection({ translations, language, theme, siteInfo }: FeaturesSectionProps) {
  const colorTheme = getColorTheme(theme);
  
  // Replace "ReviewMasters" with dynamic siteName in the title
  const featuresTitle = translations.features.title.replace(/ReviewMasters/g, siteInfo.siteName);
  
  return (
    <section 
      className="relative space-y-5 overflow-hidden rounded-2xl border p-5 shadow-xl backdrop-blur-sm sm:space-y-6 sm:rounded-3xl sm:p-8"
      style={{
        borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
        background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
        boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
      }}
    >
      {/* Subtle accent gradients - theme-based */}
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
        }}
      />
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
        }}
      />
      
      {/* Header section */}
      <div className="relative space-y-2 text-center sm:space-y-3">
        <div 
          className="inline-flex items-center gap-2 rounded-full px-3 py-1 backdrop-blur-sm sm:px-4 sm:py-1.5"
          style={{
            border: `1px solid ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
            backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.05),
          }}
        >
          <svg 
            className="h-3 w-3 sm:h-4 sm:w-4" 
            fill="none" 
            viewBox="0 0 24 24" 
            stroke="currentColor" 
            strokeWidth={2.5}
            style={{ color: colorTheme.buttonBadge }}
          >
            <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
          </svg>
          <span 
            className="text-[10px] font-semibold uppercase tracking-wider sm:text-xs"
            style={{ color: hexToRgba(colorTheme.buttonBadge, 0.9) }}
          >
            {translations.features.badge}
          </span>
        </div>
        <h2 className="text-2xl font-bold tracking-tight text-white sm:text-3xl lg:text-4xl">
          {featuresTitle}
        </h2>
        <p className="mx-auto max-w-2xl text-sm leading-relaxed text-slate-300/90 sm:text-base sm:text-white/80">
          {translations.features.subtitle}
        </p>
      </div>
      
      {/* Features grid */}
      <ul className="relative grid gap-2.5 sm:grid-cols-2 lg:grid-cols-4 sm:gap-4">
        {FEATURES_CONFIG.map((feature) => {
          const featureData = translations.features[feature.key];
          return (
            <li
              key={feature.key}
              className="group relative flex h-full min-h-[190px] flex-col overflow-hidden rounded-xl border p-5 text-center shadow-lg transition-all duration-300 ease-out hover:-translate-y-1 sm:min-h-[220px] sm:rounded-2xl sm:p-6"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
                background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
                boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
                }}
              />
              
              {/* Icon container - professional and clean */}
              <div 
                className="relative z-10 mx-auto mb-3 flex h-12 w-12 items-center justify-center rounded-xl text-white shadow-lg transition-transform duration-300 group-hover:scale-110 sm:mb-4 sm:h-14 sm:w-14"
                style={{
                  background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                  boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                }}
              >
                <svg className="h-5 w-5 sm:h-7 sm:w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                  {feature.icon}
                </svg>
              </div>
              
              {/* Content */}
              <div className="relative z-10 flex flex-1 flex-col">
                <h3 className="relative z-10 mb-2.5 text-base font-extrabold text-white sm:mb-3 sm:text-lg">
                  {featureData.title}
                </h3>
                <p className="relative z-10 text-xs leading-relaxed text-white/80 sm:text-sm">
                  {featureData.description}
                </p>
              </div>
            </li>
          );
        })}
      </ul>
    </section>
  );
}

