import type { Translations } from "@/lib/translations";
import type { SiteInfo } from "@/types/data";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface HeroSectionProps {
  readonly translations: Translations;
  readonly siteInfo: SiteInfo;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

export function HeroSection({ translations, siteInfo, language, theme }: HeroSectionProps) {
  const colorTheme = getColorTheme(theme);
  
  return (
    <section 
      className="group relative overflow-hidden rounded-2xl border px-6 py-3 text-white shadow-2xl transition-all duration-700 sm:rounded-3xl sm:px-12 sm:py-4"
      style={{
        borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
        background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
        boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
      }}
    >
      {/* Animated background gradients - theme-based */}
      <div 
        className="absolute inset-0 transition-all duration-1000 group-hover:opacity-100" 
        style={{
          background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
        }}
      />
      <div 
        className="absolute inset-0 transition-all duration-1000 group-hover:opacity-100" 
        style={{
          background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
        }}
      />
      
      {/* Animated grid pattern */}
      <div className="absolute inset-0 bg-[linear-gradient(rgba(255,255,255,0.03)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,0.03)_1px,transparent_1px)] bg-[size:30px_30px] opacity-60 transition-opacity duration-700 group-hover:opacity-80" />
      
      {/* Shimmer effect on hover */}
      <div className="absolute inset-0 -translate-x-full bg-gradient-to-r from-transparent via-white/20 to-transparent transition-transform duration-1000 group-hover:translate-x-full" />
      
      {/* Glowing border effect */}
      <div className="absolute inset-0 rounded-2xl border border-white/10 transition-all duration-700 group-hover:border-white/20 sm:rounded-3xl" />
      
      <div className="relative space-y-3 text-center sm:space-y-4">
        <span 
          className="inline-flex items-center gap-2 rounded-full px-3 py-1 text-[10px] font-semibold uppercase tracking-wider backdrop-blur-sm sm:px-4 sm:py-1.5 sm:text-xs"
          style={{
            border: `1px solid ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
            backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.05),
            color: hexToRgba(colorTheme.buttonBadge, 0.9),
          }}
        >
          <span className="relative flex h-1.5 w-1.5 sm:h-2 sm:w-2">
            <span 
              className="absolute inline-flex h-full w-full animate-ping rounded-full opacity-75" 
              style={{ backgroundColor: colorTheme.buttonBadge }}
            />
            <span 
              className="relative inline-flex h-full w-full rounded-full shadow-lg" 
              style={{
                backgroundColor: colorTheme.buttonBadge,
                boxShadow: `0 10px 15px -3px ${hexToRgba(colorTheme.buttonBadge, 0.6)}`,
              }}
            />
          </span>
          {translations.hero.badge}
        </span>
        
        {/* Sliding text below badge */}
        <div className="relative overflow-hidden py-1.5 sm:py-2">
          <div className="flex animate-slide-infinite">
            {/* First set */}
            <div className="flex shrink-0 items-center" style={{ width: 'max-content' }}>
              <span className="mr-2 shrink-0 text-[7px] text-blue-400/50 sm:mr-3 sm:text-[9px]">•</span>
              {translations.hero.slidingText.map((text, index) => (
                <span key={`first-${index}`} className="flex shrink-0 items-center">
                  <span className="shrink-0 whitespace-nowrap text-[8px] font-medium leading-tight text-slate-300/80 sm:text-[10px] sm:text-slate-300/90">
                    {text}
                  </span>
                  {index < translations.hero.slidingText.length - 1 && (
                    <span 
                      className="mx-2 shrink-0 text-[7px] sm:mx-3 sm:text-[9px]"
                      style={{ color: hexToRgba(colorTheme.buttonBadge, 0.5) }}
                    >
                      •
                    </span>
                  )}
                </span>
              ))}
              <span className="ml-4 shrink-0 sm:ml-6" />
            </div>
            {/* Duplicate for seamless loop */}
            <div className="flex shrink-0 items-center" style={{ width: 'max-content' }}>
              <span className="mr-2 shrink-0 text-[7px] text-blue-400/50 sm:mr-3 sm:text-[9px]">•</span>
              {translations.hero.slidingText.map((text, index) => (
                <span key={`second-${index}`} className="flex shrink-0 items-center">
                  <span className="shrink-0 whitespace-nowrap text-[8px] font-medium leading-tight text-slate-300/80 sm:text-[10px] sm:text-slate-300/90">
                    {text}
                  </span>
                  {index < translations.hero.slidingText.length - 1 && (
                    <span 
                      className="mx-2 shrink-0 text-[7px] sm:mx-3 sm:text-[9px]"
                      style={{ color: hexToRgba(colorTheme.buttonBadge, 0.5) }}
                    >
                      •
                    </span>
                  )}
                </span>
              ))}
              <span className="ml-4 shrink-0 sm:ml-6" />
            </div>
          </div>
        </div>
        <h1 className="relative text-2xl font-extrabold leading-[1.1] tracking-tight text-white drop-shadow-[0_2px_8px_rgba(0,0,0,0.3)] transition-all duration-500 group-hover:drop-shadow-[0_4px_16px_rgba(255,255,255,0.2)] sm:text-5xl lg:text-6xl">
          <span className="relative inline-block bg-gradient-to-r from-white via-white to-white/95 bg-clip-text text-transparent">
            {siteInfo.tagline}
          </span>
          {/* Subtle underline glow - theme-based */}
          <span 
            className="absolute -bottom-1 left-1/2 h-0.5 w-0 -translate-x-1/2 transition-all duration-700 group-hover:w-3/4 sm:-bottom-1.5 sm:h-1"
            style={{
              background: `linear-gradient(to right, transparent, ${colorTheme.buttonBadge}, transparent)`,
            }}
          />
        </h1>
      </div>
      
      {/* Floating particles effect - theme-based */}
      <div 
        className="absolute top-1/4 left-1/4 h-1 w-1 rounded-full opacity-60 animate-pulse" 
        style={{ backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.5) }}
      />
      <div 
        className="absolute top-1/3 right-1/3 h-1.5 w-1.5 rounded-full opacity-50 animate-pulse delay-300" 
        style={{ backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.4) }}
      />
      <div 
        className="absolute bottom-1/4 right-1/4 h-1 w-1 rounded-full opacity-60 animate-pulse delay-700" 
        style={{ backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.4) }}
      />
    </section>
  );
}

