import type { Partner, SiteInfo } from "@/types/data";
import { BASE_URL } from "./metadata";

/**
 * Generates Organization structured data
 */
export function generateOrganizationLd(siteInfo: SiteInfo) {
  return {
    "@context": "https://schema.org",
    "@type": "Organization",
    name: siteInfo.siteName,
    url: `${BASE_URL}/`,
    logo: `${BASE_URL}${siteInfo.siteLogo}`,
    email: siteInfo.email,
    address: {
      "@type": "PostalAddress",
      addressLocality: siteInfo.address,
      addressCountry: "UK",
    },
    contactPoint: [
      {
        "@type": "ContactPoint",
        telephone: siteInfo.phone,
        contactType: "customer support",
        areaServed: "GB",
        availableLanguage: ["Portuguese", "Português"],
      },
    ],
  } as const;
}

/**
 * Generates ItemList structured data for partners
 * Only includes licensed partners to prevent indexing of unlicensed partners
 */
export function generateItemListLd(licensedPartners: readonly Partner[], siteName: string) {
  return {
    "@context": "https://schema.org",
    "@type": "ItemList",
    name: `${siteName} Partner Offers`,
    itemListElement: licensedPartners.map((partner, index) => ({
      "@type": "ListItem",
      position: index + 1,
      name: partner.name,
      description: partner.welcomeBonus,
      url: `${BASE_URL}/#${partner.name.toLowerCase().replace(/[^a-z0-9]+/g, "-")}`,
    })),
  } as const;
}

/**
 * Generates Review structured data for featured partner
 */
export function generateReviewLd(featuredPartner: Partner | undefined, siteName: string) {
  if (!featuredPartner) return null;

  return {
    "@context": "https://schema.org",
    "@type": "Review",
    itemReviewed: {
      "@type": "Organization",
      name: featuredPartner.name,
      url: `${BASE_URL}/`,
    },
    reviewRating: {
      "@type": "Rating",
      ratingValue: "4.7",
      bestRating: "5",
    },
    author: {
      "@type": "Organization",
      name: siteName,
    },
  } as const;
}

